# Form Checkbox Inputs

> For cross browser consistency, `<b-form-checkbox-group>` and `<b-form-checkbox>` use
Bootstrap's custom checkbox input to replace the browser default checkbox input. It
is built on top of semantic and accessible markup, so it is a solid replacement for
the default checkbox input.

**Example 1:** Single checkbox
```html
<template>
  <div>
    <b-form-checkbox id="checkbox1"
                     v-model="status"
                     value="accepted"
                     unchecked-value="not_accepted">
      I accept the terms and use
    </b-form-checkbox>
    <div>State: <strong>{{status}}</strong></div>
  </div>
</template>

<script>
export default {
    data: {
        status: 'not_accepted'
    }
}
</script>

<!-- form-checkbox-1.vue -->
```

**Example 2:** Multiple choice checkboxes
```html
<template>
  <div>
    <h5>Using options array:</h5>
    <b-form-checkbox-group id="checkboxes1" name="flavour1" v-model="selected" :options="options">
    </b-form-checkbox-group>

    <h5 class="mt-3">Using sub-components:</h5>
    <b-form-checkbox-group id="checkboxes2" name="flavour2" v-model="selected">
      <b-form-checkbox value="orange">Orange</b-form-checkbox>
      <b-form-checkbox value="apple">Apple</b-form-checkbox>
      <b-form-checkbox value="pineapple">Pineapple</b-form-checkbox>
      <b-form-checkbox value="grape">Grape</b-form-checkbox>
    </b-form-checkbox-group>
    <hr>
    <div>Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data: {
      selected: [], // Must be an array reference!
      options: [
        {text: 'Orange', value: 'orange'},
        {text: 'Apple', value: 'apple'},
        {text: 'Pineapple', value: 'pineapple'},
        {text: 'Grape', value: 'grape'}
      ]
    }
  };
</script>

<!-- form-checkbox-2.vue -->
```

Feel free to mix and match `options` prop and `<b-form-checkbox>` in `<b-form-checkbox-group>`.
Manually placed `<b-form-checkbox>` inputs will appear _below_ any checkbox inputs generated by
the `options` prop. To have them apper _above_ the inputs generated by `options`, place them
in the named slot `first`.


## Options
Please see options in [`<b-form-select>`](/docs/components/form-select) docs for details on passing
options (value array) to `<b-form-checkboxes>`.

## Inline and Stacked checkboxes
`<b-form-checkbox>` components render as inline elements by default. Set the prop `stacked` on
`<b-form-checkbox-group>` to place each form control is on a separate line.

```html
<template>
  <div>
    <h5>Inline checkboxes (default)</h5>
    <b-form-checkbox-group v-model="selected" name="flavour1" :options="options">
    </b-form-checkbox-group>

    <h5 class="mt-3">Stacked checkboxes</h5>
    <b-form-checkbox-group stacked v-model="selected" name="flavour2" :options="options">
    </b-form-checkbox-group>
  </div>
</template>

<script>
  export default {
    data: {
      selected: [], // Must be an array reference!
      options: [
        {text: 'Orange', value: 'orange'},
        {text: 'Apple', value: 'apple'},
        {text: 'Pineapple', value: 'pineapple'},
        {text: 'Grape', value: 'grape'}
      ]
    }
  };
</script>

<!-- form-checkbox-stacked-1.vue -->
```

## Value(s)
By default, `<b-form-checkbox>` value will be `true` when checked and `false` when unchecked.
You can customize the checked and unchecked values by specifying the `value` and `unchecked-value`
properties.

`v-model` binds to the `checked` property. When you have multiple checkboxes that bind to a
single data state variable, you **must** provide an array reference `[]` to your `v-model`!

Note that when `v-model` is bound to multiple checkboxes (i.e an array ref), the
`unchecked-value` is **not used**. Only the value(s) of the checked chcekboxes will
be returned in the `v-model` bound array. You should provide unique values for each
checkbox's `value` prop.


### Multiple checkboxes and accessibility
When binding multiple checkboxes together, you should set the `name` prop to the same
value for all `<b-formcheckbox>`s in the group individually or via the `name` prop
of `<b-form-checkbox-group>`. This will inform users of assitive technologies that the
checkboxes are related.

Whenever using multple checkboxes, it is recommended that the `<b-form-check-group>`
be placed in a [`<b-form-group>`](/docs/components/form-group) component to
associate a `<label>` with the entire group of checkboxes.

## Button style checkboxes
Render checkboxes with the look of a button by setting the prop `buttons` on `<b-form-checkbox-group>`.
Change the button variant by setting the `button-variant` prop to one of the standard Bootstrap
button variants (see [`<b-button>`](/docs/components/button) for supported variants). The
default `button-variant` is `secondary`.

Button style checkboxes will have the class `.active` automatically applied to the label
when they are in the _checked_ state.

```html
<template>
  <div>
    <h5>button style checkboxes</h5>
    <b-form-checkbox-group buttons v-model="selected" name="butons1" :options="options">
    </b-form-checkbox-group>

    <h5 class="mt-3">
      button style checkboxes with variant <code>primary</code> and large buttons
    </h5>
    <b-form-checkbox-group v-model="selected"
                           buttons
                           button-variant="primary"
                           size="lg"
                           name="buttons2"
                           :options="options">
    </b-form-checkbox-group>

    <h5 class="mt-3">Stacked (vertical) button style checkboxes</h5>
    <b-form-checkbox-group buttons v-model="selected" stacked :options="options">
    </b-form-checkbox-group>
  </div>
</template>

<script>
  export default {
    data: {
      selected: [], // Must be an array reference!
      options: [
        {text: 'Orange', value: 'orange'},
        {text: 'Apple', value: 'apple'},
        {text: 'Pineapple', value: 'pineapple'},
        {text: 'Grape', value: 'grape'}
      ]
    }
  };
</script>

<!-- form-checkboxs-buttons.vue -->
```


## Non custom check inputs (plain)
You can have `<b-form-checkbox-group>` or `<b-form-checkbox>` render a browser native
chechbox input by setting the `plain` prop.

```html
<template>
  <div>
    <h5>Plain inline checkboxes</h5>
    <b-form-checkbox-group plain v-model="selected" :options="options" />

    <h5 class="mt-3">Plain stacked checkboxes</h5>
    <b-form-checkbox-group plain stacked v-model="selected" :options="options" />
  </div>
</template>

<script>
  export default {
    data: {
      selected: [], // Must be an array reference!
      options: [
        {text: 'Orange', value: 'orange'},
        {text: 'Apple', value: 'apple'},
        {text: 'Pineapple', value: 'pineapple'},
        {text: 'Grape', value: 'grape'}
      ]
    }
  };
</script>

<!-- form-checkbox-plain-1.vue -->
```

**Note:** The `plain` prop has no effect when `button` is set.


## Contextual states
Bootstrap includes validation styles for `valid` and `invalid` states
on most form controls.

Generally speaking, you’ll want to use a particular state for specific types of feedback:
- `'invalid'` is great for when there’s a blocking or required field. A user must fill in
this field properly to submit the form.
- `'valid'` is ideal for situations when you have per-field validation throughout a form
and want to encourage a user through the rest of the fields.
- `null` Displays no validation state

To apply one of the contextual state icons on `<b-form-checkbox>`, set the `state` prop
to `'invalid'`, `'valid'`, or `null`.

**Note:** Contextual states are **not** supported when in button mode.


## Indeterminate (tri-state) support
Normally a checkbox input can only have two states: _checked_ or _unchecked_. They can
have any value, but they either submit that value (_checked_) or don't (_unchecked_) with
a form submission (although Bootstrap-Vue allows a value for the _unchecked_ state)

_Visually_, there are actually three states a checkbox can be in: _checked_,
_unchecked_, or **_indeterminate_**.

The _indeterminate_ state is **visual only**. The checkbox is still either checked or
unchecked as a value. That means the visual indeterminate state masks the real value
of the checkbox, so that better make sense in your UI!

`<b-form-checkbox>` supports setting this visual indeterminate state via the `indeterminate`
prop (defaults to `false`). Clicking the checkbox will clear its indeterminate state.
The `indeterminate` prop can be synced to the checkboxe's state by v-binding the
`indeterminate` prop with the `.sync` modifier.

**Note:** indeterminate is not supported in button mode, nor is it supported in
`<b-form-checkbox-group>` (multiple checkboxes).

**Single Indeterminate checkbox:**
```html
<template>
  <div>
    <b-form-checkbox v-model="checked" :indeterminate.sync="indeterminate">
      Click me to see what happens
    </b-form-checkbox>

    <div class="mt-3">
      Checked: <strong>{{ checked }}</strong><br>
      Indeterminate: <strong>{{ indeterminate }}</strong>
    </div>
    <b-btn @click="toggleIndeterminate">Toggle Indeterminate State</b-btn>
  </div>
</template>

<script>
export default {
    data: {
        checked: true,
        indeterminate: true
    },
    methods: {
        toggleIndeterminate() {
            this.indeterminate = !this.indeterminate;
        }
    }
}
</script>

<!-- form-checkbox-indeterminate-1.vue -->
```

**Indeterminate checkbox use-case:**
```html
<template>
  <b-card>
    <b-form-checkbox v-model="allSelected"
                     :indeterminate="indeterminate"
                     aria-describedby="flavours"
                     aria-controls="flavours"
                     @change="toggleAll"
     >
      {{ allSelected ? 'Un-select' : 'Select' }}
      All Flavors
    </b-form-checkbox>
    <b-form-checkbox-group id="flavors"
                           stacked
                           v-model="selected"
                           name="flavs"
                           :options="flavours"
                           class="ml-3"
                           aria-label="Individual flavours"
    ></b-form-checkbox-group>
    <p aria-live="polite">
      Selected: <strong>{{ selected }}</strong><br>
      All Selected: <strong>{{ allSelected }}</strong><br>
      Indeterminate: <strong>{{ indeterminate }}</strong><br>
    </p>
  </b-card>
</template>

<script>
export default {
    data: {
        flavours: ['Orange', 'Grape', 'Apple', 'Lime', 'Very Berry'],
        selected: [],
        allSelected: false,
        indeterminate: false
    },
    methods: {
       toggleAll(checked) {
          this.selected = checked ? this.flavours.slice() : [];
       }
    },
    watch: {
        selected(newVal, oldVal) {
            // Handle changes in individual flavour checkboxes
            if (newVal.length === 0) {
                this.indeterminate = false;
                this.allSelected = false;
            } else if (newVal.length === this.flavours.length) {
                this.indeterminate = false;
                this.allSelected = true;
            } else {
                this.indeterminate = true;
                this.allSelected = false;
            }
        }
    }
}
</script>

<!-- form-checkbox-indeterminate-2.vue -->
```

**Note:** indeterminate is not supported in `button` mode, nor in multiple checkbox mode.

### Indeterminate state and accessibility
Not all screen readers will convey the indeterminate state to screen reader users.
So it is recommended to provide some form of textual feedback to the user (possibly
by via the `.sr-only` class) if the indeterminate state has special contextual
meaning in your application.


## Checkbox component aliases
- `<b-form-checkbox-group>` can be used by the shorter aliases `<b-checkbox-group>` and `<b-check-group>`.
- `<b-form-checkbox>` can be used by thes shorter aliases `<b-checkbox>` and `<b-check>`.


## Component Reference
