var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

require('../../../src/stylus/components/_text-fields.styl');
require('../../../src/stylus/components/_input-groups.styl');
require('../../../src/stylus/components/_select.styl');

// Components
import VCard from '../VCard';
import VCheckbox from '../VCheckbox';
import { VList, VListTile, VListTileAction, VListTileContent, VListTileTitle } from '../VList';
import VMenu from '../VMenu';
import VBtn from '../VBtn';

// Mixins
import Colorable from '../../mixins/colorable';
import Dependent from '../../mixins/dependent';
import Filterable from '../../mixins/filterable';
import Input from '../../mixins/input';
import Maskable from '../../mixins/maskable';

// Component level mixins
import Autocomplete from './mixins/select-autocomplete';
import Generators from './mixins/select-generators';

// Directives
import ClickOutside from '../../directives/click-outside';

// Helpers
import { getObjectValueByPath } from '../../util/helpers';

export default {
  name: 'v-select',

  inheritAttrs: false,

  components: {
    VCard: VCard,
    VCheckbox: VCheckbox,
    VList: VList,
    VListTile: VListTile,
    VListTileAction: VListTileAction,
    VListTileContent: VListTileContent,
    VListTileTitle: VListTileTitle,
    VMenu: VMenu,
    VBtn: VBtn
  },

  directives: {
    ClickOutside: ClickOutside
  },

  mixins: [Autocomplete, Colorable, Dependent, Filterable, Generators, Input, Maskable],

  data: function data() {
    return {
      cachedItems: [],
      content: {},
      inputValue: (this.multiple || this.tags) && !this.value ? [] : this.value,
      isBooted: false,
      lastItem: 20,
      lazySearch: null,
      isActive: false,
      menuIsActive: false,
      searchTimeout: null,
      selectedIndex: -1,
      selectedItems: [],
      shouldBreak: false
    };
  },


  props: {
    appendIcon: {
      type: String,
      default: 'arrow_drop_down'
    },
    appendIconCb: Function,
    auto: Boolean,
    autocomplete: Boolean,
    cacheItems: Boolean,
    chips: Boolean,
    clearable: Boolean,
    color: {
      type: String,
      default: 'primary'
    },
    combobox: Boolean,
    debounceSearch: {
      type: [Number, String],
      default: 200
    },
    browserAutocomplete: {
      type: String,
      default: 'on'
    },
    items: {
      type: Array,
      default: function _default() {
        return [];
      }
    },
    itemAvatar: {
      type: String,
      default: 'avatar'
    },
    itemText: {
      type: String,
      default: 'text'
    },
    itemValue: {
      type: String,
      default: 'value'
    },
    itemDisabled: {
      type: String,
      default: 'disabled'
    },
    maxHeight: {
      type: [Number, String],
      default: 300
    },
    minWidth: {
      type: [Boolean, Number, String],
      default: false
    },
    multiple: Boolean,
    multiLine: Boolean,
    solo: Boolean,
    searchInput: {
      default: null
    },
    singleLine: Boolean,
    tags: Boolean,
    returnObject: Boolean,
    overflow: Boolean,
    segmented: Boolean,
    editable: Boolean
  },

  computed: {
    classes: function classes() {
      var classes = {
        'input-group--text-field input-group--select': true,
        'input-group--auto': this.auto,
        'input-group--overflow': this.overflow,
        'input-group--segmented': this.segmented,
        'input-group--editable': this.editable,
        'input-group--autocomplete': this.isAutocomplete,
        'input-group--single-line': this.singleLine || this.isDropdown,
        'input-group--multi-line': this.multiLine,
        'input-group--chips': this.chips,
        'input-group--solo': this.solo,
        'input-group--multiple': this.multiple
      };

      if (this.hasError) {
        classes['error--text'] = true;
      } else {
        return this.addTextColorClassChecks(classes);
      }

      return classes;
    },
    computedContentClass: function computedContentClass() {
      var children = ['menu__content--select', this.auto ? 'menu__content--auto' : '', this.isDropdown ? 'menu__content--dropdown' : '', this.isAutocomplete ? 'menu__content--autocomplete' : ''];

      return children.join(' ');
    },
    computedItems: function computedItems() {
      return this.filterDuplicates(this.cachedItems.concat(this.items));
    },

    /**
     * The range of the current input text
     *
     * @return {Number}
     */
    currentRange: function currentRange() {
      return this.getText(this.selectedItem || '').length;
    },
    filteredItems: function filteredItems() {
      // If we are not actively filtering
      // Show all available items
      var items = this.isNotFiltering ? this.computedItems : this.filterSearch();

      return !this.auto ? items.slice(0, this.lastItem) : items;
    },
    hideSelections: function hideSelections() {
      return this.isAutocomplete && !this.isMultiple && this.isFocused && !this.chips;
    },
    isNotFiltering: function isNotFiltering() {
      return this.isAutocomplete && this.isDirty && this.searchValue === this.getText(this.selectedItem);
    },
    isAutocomplete: function isAutocomplete() {
      return this.autocomplete || this.editable || this.tags || this.combobox;
    },
    isDirty: function isDirty() {
      return this.selectedItems.length > 0;
    },
    isDropdown: function isDropdown() {
      return this.segmented || this.overflow || this.editable || this.solo;
    },
    isMultiple: function isMultiple() {
      return this.multiple || this.tags;
    },

    searchValue: {
      get: function get() {
        return this.lazySearch;
      },
      set: function set(val) {
        var _this = this;

        if (!this.isAutocomplete || this.selectedIndex > -1) return;

        this.lazySearch = val;

        clearTimeout(this.searchTimeout);

        this.searchTimeout = setTimeout(function () {
          _this.$emit('update:searchInput', val);
        }, this.debounceSearch);
      }
    },
    selectedItem: function selectedItem() {
      var _this2 = this;

      if (this.isMultiple) return null;

      return this.selectedItems.find(function (i) {
        return _this2.getValue(i) === _this2.getValue(_this2.inputValue);
      }) || null;
    }
  },

  watch: {
    inputValue: function inputValue(val) {
      // Populate selected items
      this.genSelectedItems(val);

      this.$emit('input', val);

      // When inputValue is changed
      // and combobox is true set
      // menu property to false
      if (this.combobox) this.menuIsActive = false;
    },
    isActive: function isActive(val) {
      if (!val) {
        this.searchValue = null;
        this.menuIsActive = false;
        this.isFocused = false;
        this.selectedIndex = -1;
      } else {
        this.searchValue = this.getText(this.selectedItem);
      }

      // this.lastItem += !val ? 20 : 0
    },
    isBooted: function isBooted() {
      var _this3 = this;

      this.$nextTick(function () {
        if (_this3.content && _this3.content.addEventListener) {
          _this3.content.addEventListener('scroll', _this3.onScroll, false);
        }
      });
    },
    isFocused: function isFocused(val) {
      var _this4 = this;

      // When focusing the input
      // re-set the caret position
      if (this.isAutocomplete && !this.mask && !this.isMultiple && val) {
        this.setCaretPosition(this.currentRange);
        this.shouldBreak && this.$nextTick(function () {
          _this4.$refs.input.scrollLeft = _this4.$refs.input.scrollWidth;
        });
      }
    },
    items: function items(val) {
      var _this5 = this;

      if (this.cacheItems) {
        this.cachedItems = this.filterDuplicates(this.cachedItems.concat(val));
      }

      this.$refs.menu.listIndex = -1;

      this.searchValue && this.$nextTick(function () {
        _this5.$refs.menu && (_this5.$refs.menu.listIndex = 0);
      });

      this.genSelectedItems();
    },
    menuIsActive: function menuIsActive(val) {
      if (!val) return;

      this.isBooted = true;
      this.isActive = true;
    },
    isMultiple: function isMultiple(val) {
      this.inputValue = val ? [] : null;
    },
    searchInput: function searchInput(val) {
      this.searchValue = val;
    },
    searchValue: function searchValue(val) {
      var _this6 = this;

      // Wrap input to next line if overflowing
      if (this.$refs.input.scrollWidth > this.$refs.input.clientWidth) {
        this.shouldBreak = true;
        this.$nextTick(this.$refs.menu.updateDimensions);
      } else if (val === null) {
        this.shouldBreak = false;
      }

      // Activate menu if inactive and searching
      if (this.isActive && !this.menuIsActive && val !== this.getValue(this.selectedItem)) {
        this.menuIsActive = true;
      }

      this.$refs.menu.listIndex = null;

      this.$nextTick(function () {
        _this6.$refs.menu && (_this6.$refs.menu.listIndex = val ? 0 : -1);
      });
    },
    selectedItems: function selectedItems() {
      clearTimeout(this.searchTimeout);

      if (this.isAutocomplete) {
        this.$nextTick(this.$refs.menu.updateDimensions);
      }
    },
    value: function value(val) {
      this.inputValue = val;
      this.validate();
    }
  },

  mounted: function mounted() {
    // If instance is being destroyed
    // do not run mounted functions
    if (this._isDestroyed) return;

    // Evaluate the selected items immediately
    // to avoid a unnecessary label transition
    this.genSelectedItems();

    this.content = this.$refs.menu.$refs.content;
  },
  beforeDestroy: function beforeDestroy() {
    if (this.isBooted) {
      if (this.content) {
        this.content.removeEventListener('scroll', this.onScroll, false);
      }
    }
  },


  methods: {
    blur: function blur() {
      var _this7 = this;

      this.$emit('blur');
      if (this.isAutocomplete && this.$refs.input) this.$refs.input.blur();
      this.$nextTick(function () {
        return _this7.isActive = false;
      });
    },
    changeSelectedIndex: function changeSelectedIndex(keyCode) {
      // backspace, left, right, delete
      if (![8, 37, 39, 46].includes(keyCode)) return;

      var indexes = this.selectedItems.length - 1;

      if (keyCode === 37) {
        // Left arrow
        this.selectedIndex = this.selectedIndex === -1 ? indexes : this.selectedIndex - 1;
      } else if (keyCode === 39) {
        // Right arrow
        this.selectedIndex = this.selectedIndex >= indexes ? -1 : this.selectedIndex + 1;
      } else if (this.selectedIndex === -1) {
        this.selectedIndex = indexes;
        return;
      }

      // backspace/delete
      if ([8, 46].includes(keyCode)) {
        var newIndex = this.selectedIndex === indexes ? this.selectedIndex - 1 : this.selectedItems[this.selectedIndex + 1] ? this.selectedIndex : -1;

        this.selectItem(this.selectedItems[this.selectedIndex]);
        this.selectedIndex = newIndex;
      }
    },
    compareObjects: function compareObjects(a, b) {
      var aProps = Object.keys(a);
      var bProps = Object.keys(b);

      if (aProps.length !== bProps.length) return false;

      for (var i = 0, length = aProps.length; i < length; i++) {
        var propName = aProps[i];

        if (a[propName] !== b[propName]) return false;
      }

      return true;
    },
    filterDuplicates: function filterDuplicates(arr) {
      var values = arr.map(this.getValue);
      return arr.filter(function (el, i) {
        return i === values.indexOf(values[i]);
      });
    },
    focus: function focus() {
      this.isActive = true;
      this.isFocused = true;

      if (this.$refs.input && this.isAutocomplete) {
        this.$refs.input.focus();
      } else {
        this.$el.focus();
      }

      this.$emit('focus');
    },
    genDirectives: function genDirectives() {
      var _this8 = this;

      return [{
        name: 'click-outside',
        value: function value(e) {
          return _this8.$refs.menu && !_this8.$refs.menu.$refs.content.contains(e.target);
        }
      }];
    },
    genListeners: function genListeners() {
      var _this9 = this;

      var listeners = Object.assign({}, this.$listeners);
      delete listeners.input;

      return _extends({}, listeners, {
        click: function click() {
          if (_this9.disabled || _this9.readonly) return;
          _this9.showMenuItems();
          _this9.selectedIndex = -1;
        },
        focus: function focus() {
          if (_this9.disabled || _this9.readonly) return;

          !_this9.isFocused && _this9.focus();
        },
        keydown: this.onKeyDown // Located in mixins/select-autocomplete.js
      });
    },
    genLabel: function genLabel() {
      var singleLine = this.singleLine || this.isDropdown;
      if (singleLine && this.isDirty || singleLine && this.isFocused && this.searchValue) return null;

      var data = {};

      if (this.id) data.attrs = { for: this.id };

      return this.$createElement('label', data, this.$slots.label || this.label);
    },
    getPropertyFromItem: function getPropertyFromItem(item, field) {
      if (item !== Object(item)) return item;

      var value = getObjectValueByPath(item, field);

      return typeof value === 'undefined' ? item : value;
    },
    genSelectedItems: function genSelectedItems() {
      var _this10 = this;

      var val = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.inputValue;

      // If we are using tags, don't filter results
      if (this.tags) return this.selectedItems = val;

      // Combobox is the single version
      // of a taggable select element
      if (this.combobox) return this.selectedItems = val ? [val] : [];

      var selectedItems = this.computedItems.filter(function (i) {
        if (!_this10.isMultiple) {
          return _this10.getValue(i) === _this10.getValue(val);
        } else {
          // Always return Boolean
          return val.find(function (j) {
            var a = _this10.getValue(j);
            var b = _this10.getValue(i);

            if (a !== Object(a)) return a === b;

            return _this10.compareObjects(a, b);
          }) !== undefined;
        }
      });

      if (!selectedItems.length && val != null && this.tags) {
        selectedItems = Array.isArray(val) ? val : [val];
      }

      this.selectedItems = selectedItems;
    },
    getText: function getText(item) {
      return this.getPropertyFromItem(item, this.itemText);
    },
    getValue: function getValue(item) {
      return this.getPropertyFromItem(item, this.itemValue);
    },
    clearableCallback: function clearableCallback() {
      var inputValue = this.isMultiple ? [] : null;

      this.inputValue = inputValue;
      this.searchValue = null;
      this.$emit('change', inputValue);
      this.genSelectedItems();
      this.showMenu();
    },
    showMenu: function showMenu() {
      this.showMenuItems();
      this.isAutocomplete && this.focus();
    },
    onScroll: function onScroll() {
      var _this11 = this;

      if (!this.isActive) {
        requestAnimationFrame(function () {
          return _this11.content.scrollTop = 0;
        });
      } else {
        if (this.lastItem >= this.computedItems.length) return;

        var showMoreItems = this.content.scrollHeight - (this.content.scrollTop + this.content.clientHeight) < 200;

        if (showMoreItems) {
          this.lastItem += 20;
        }
      }
    },
    selectItem: function selectItem(item) {
      var _this12 = this;

      if (!this.isMultiple) {
        this.inputValue = this.returnObject ? item : this.getValue(item);
        this.selectedItems = [item];
      } else {
        var selectedItems = [];
        var inputValue = this.inputValue.slice();
        var i = this.inputValue.findIndex(function (i) {
          var a = _this12.getValue(i);
          var b = _this12.getValue(item);

          if (a !== Object(a)) return a === b;

          return _this12.compareObjects(a, b);
        });

        i !== -1 && inputValue.splice(i, 1) || inputValue.push(item);
        this.inputValue = inputValue.map(function (i) {
          selectedItems.push(i);
          return _this12.returnObject ? i : _this12.getValue(i);
        });

        this.selectedItems = selectedItems;
      }

      this.searchValue = !this.isMultiple || this.chips ? this.getText(this.selectedItem) : '';

      this.$emit('change', this.inputValue);

      // List tile will re-render, reset index to
      // maintain highlighting
      var savedIndex = this.$refs.menu.listIndex;
      this.$refs.menu.listIndex = -1;

      // After selecting an item
      // refocus the input and
      // reset the caret pos
      this.$nextTick(function () {
        _this12.focus();
        _this12.setCaretPosition(_this12.currentRange);
        _this12.$refs.menu && (_this12.$refs.menu.listIndex = savedIndex);
      });
    },
    showMenuItems: function showMenuItems() {
      this.isActive = true;
      this.menuIsActive = true;
      this.chips && (this.$refs.menu.listIndex = -1);
    }
  },

  render: function render(h) {
    var _this13 = this;

    var data = {
      attrs: _extends({
        tabindex: this.isAutocomplete || this.disabled ? -1 : this.tabindex
      }, this.isAutocomplete ? null : this.$attrs, {
        role: this.isAutocomplete ? null : 'combobox'
      })
    };

    if (!this.isAutocomplete) {
      data.on = this.genListeners();
      data.directives = this.genDirectives();
    } else {
      data.on = {
        click: function click() {
          if (_this13.disabled || _this13.readonly) return;

          // Workaround for clicking select
          // when using autocomplete
          // and click doesn't target the input
          setTimeout(function () {
            if (_this13.menuIsActive) return;

            _this13.focus();
            _this13.menuIsActive = true;
          }, 100);
        }
      };
    }

    return this.genInputGroup([this.genSelectionsAndSearch(), this.genMenu()], data, function () {
      return _this13.showMenu();
    });
  }
};